;;; @(#) cua-lite.el -- a light-weight, extendable & toggle-able CUA emulator
;;; @(#) $Id: cua-lite.el,v 1.3 2001/07/16 03:31:53 jcasa Exp $

;; This file is not part of Emacs

;; Copyright (C) 2001 by Joseph L. Casadonte Jr.
;; Author:          Joe Casadonte (emacs@northbound-train.com)
;; Maintainer:      Joe Casadonte (emacs@northbound-train.com)
;; Created:         May 27, 2001
;; Keywords:        CUA emulator
;; Latest Version:  http://www.northbound-train.com/emacs.html

;; COPYRIGHT NOTICE

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:
;;
;;  cua-lite is a CUA emulator package for Emacs.  There are three
;;  things that make this package different from the others already
;;  out there:
;;
;;  1) it is a quasi-minor mode, meaning that it can be turned on &
;;     off easily (though only globally; hence the "quasi-")
;;  2) it is easily extendable
;;  3) it does NOT attempt to override the existing use of <Ctrl><X>
;;     and <Ctrl><C> (use CUA-mode for that)

;;; Installation:
;;
;;  There are two ways to install this package.  The first is to
;;  install it and load it immediately.  This is the appropriate
;;  method if you know you want to load it all of the time.
;;
;;  The second method is to use a bootstrap package,
;;  `win-bootstrap'.  This package will set up an autoload
;;  statement for win, and (optionally) define a hotkey to toggle
;;  the package on and off.  This is to encourage people to have the
;;  package available for use by others even if they themselves do not
;;  wish to use it, while at the same time utilizing almost no
;;  resources.  See the bootstrap file for more details.
;;
;;  To load and enable each time (method #1), put this file on your
;;  Emacs-Lisp load path and add the following to your ~/.emacs
;;  startup file:
;;
;;     (require 'win)
;;     (win 1)
;;
;;  To add a directory to your load-path, use something like the following:
;;
;;      (add-to-list 'load-path (expand-file-name "/some/load/path"))

;;; Usage:
;;
;;  M-x `win'
;;     Toggles the package on and off.  With optional parameter it
;;     turns win on iff the arg is a positive integer.
;;
;;     Generally speaking, turning win on will bind certain
;;     movement commands to keys and enable some CUA-like features
;;     within Emacs (see specifics below).  Movement commands are
;;     functions that move point and movement keys are the keys bound
;;     to those commands.  Simple examples include the up arrow and
;;     the down arrow, which are bound to `previous-line' and
;;     `next-line', respectively.  What's special about them in
;;     win, though, is that when the Shift key is held down while
;;     pressing the key bound to the movement command, the current
;;     selection is extended to cover the area traversed by the
;;     movement command.  For example, say point is right here -><-
;;     and I hit the 'home' key.  Normally (at least in a CUA
;;     emulation mode) point would move to the beginning of the line,
;;     column 0.  If the Shift key were held down while I hit the
;;     'home' key, though, then everything from the current point to
;;     the beginning of the line would be selected.  Hitting a
;;     movement key while a selection is activated, but without the
;;     Shift key being pressed, will de-activate the selection.  It's
;;     all pretty standard behavior in a CUA application (though I'm
;;     not quite sure if CUA defines this behavior explicitly or not).
;;
;;     Non-movement keys bind common Emacs commands to CUA-standard
;;     keys.  Examples would be <Ctrl><S> for Save (`save-buffer') and
;;     <Ctrl><A> for Select All (`mark-whole-buffer').  None of these
;;     commands work on or are affected by the current selection,
;;     though.
;;
;;     One set of CUA keys that are explicitly NOT bound are the Cut,
;;     Copy & Paste key combinations, <Ctrl><X>, <Ctrl><C> and
;;     <Ctrl><V>, respectively.  This is because Emacs has a great
;;     deal of function bound to C-x and C-c by default, and CUA
;;     provides another mechanism for Cut, Copy & Paste
;;     (<Shift><Delete>, <Control><Insert> and <Shift><Insert>,
;;     respectively).  If that particular set of key bindings is of
;;     paramount importance, I suggest using CUA-mode
;;     (http://hjem.get2net.dk/storm/emacs/).
;;
;;     When win is enabled, the following keys are bound (some
;;     are optional):
;;
;;       Movement keys (pressing Shift with any of these extends the
;;       current selection):
;;         o <left>, <right>, <up>, <down> - standard definitions
;;         o C-<right>, C-<left> - forward & backward word
;;         o C-<up>, C-<down> - forward & backward paragraph
;;         o M-<up>, M-<down> - forward & backward paragraph
;;           see `win-use-simplified-paragraph-movement'
;;         o <prior>, <next> - page up & down (scroll down & up, respectively)
;;         o C-<prior>, C-<next> - top & bottom of page
;;           see `win-use-page-movement-keys'
;;         o <home>, <end> - beginning & end of line
;;           see `win-use-home-key-bounce' & `win-use-end-key-bounce'
;;         o C-<home>, C-<end> - beginning & end of buffer
;;
;;       Non-movement keys:
;;         o C-<backspace> - delete word backward
;;           see `win-use-backward-delete-word'
;;         o C-a - Select All (`mark-whole-buffer')
;;         o C-f - Find (`isearch-forward')
;;         o M-f - Find Backwards (`isearch-backward')
;;         o C-o - Open (`find-file')
;;         o C-r - Replace (`replace-string')
;;         o C-s - Save (`save-buffer')
;;         o M-s - Save As (`write-file')
;;         o C-w - Close (`kill-buffer')
;;         o C-z - Undo (`undo' / `advertised-undo')
;;
;;     In addition, certain CUA-like features of Emacs are enabled.
;;     This is accomplished by modifying the values of the following
;;     variables or calling the following functions (which see):
;;         o `mark-active' - see `win-keep-current-mark'
;;         o `truncate-lines' - see `win-use-hscroll-mode'
;;         o `hscroll-global-mode' - see `win-use-hscroll-mode'
;;         o `blinking-cursor-mode' - see `win-use-blinking-cursor'
;;         o `bar-cursor-mode' - see `win-use-bar-cursor-mode'
;;         o `transient-mark-mode'
;;         o `mark-even-if-inactive' - see `win-disable-inactive-regions'
;;         o `delete-selection-mode' - see `win-use-delete-selection-mode'
;;
;;     Both of the above-mentioned keybinding concepts (movement keys
;;     and non-movement keys) can be extended to your own creations
;;     via one of four simple functions.  Using these functions to
;;     bind your keys ensures that when win is disabled, your
;;     key-bindings revert back to Emacs-normal, and when win is
;;     re-enabled, your keys come back with it.
;;
;;     I recommend that these functions be called in the hook
;;     `win-bind-keys-hook' (this hook can be customized).
;;     Please see the individual functions for more details on how
;;     they're used.  Here's an example of what I have:
;;
;;       (defun my-win-keys ()
;;         "Bunch of stuff to run for win when keys are bound."
;;         (win-bind-motion-key "C-=" 'joc-bounce-sexp nil)
;;         (win-bind-motion-key "C-+" 'joc-bounce-sexp t)
;;
;;         (if (or (eq window-system 'w32) (eq window-system win32))
;;       	  (win-bind-key-simple "C-p" 'joc-print-buffer-or-region))
;;
;;         (win-save-keystroke-for-restoration "C-<kp-enter>")
;;         (global-set-key (read-kbd-macro "C-<kp-enter>")
;;       				  '(lambda ()
;;       					 "Join current line to next line, deleting white space."
;;       					 (interactive)
;;       					 (delete-indentation 1)))
;;
;;         ;; camelCase mode - http://www.ai.mit.edu/people/caroma/tools/
;;         (win-bind-both-motion-keys "M-<right>" 'camelCase-forward-word)
;;         (win-bind-both-motion-keys "M-<left>" 'camelCase-backward-word)
;;       )
;;
;;       ;; DO NOT EDIT THESE MANUALLY
;;       (custom-set-variables
;;              ....
;;        '(win-bind-keys-hook (quote (my-win-keys)))
;;              ....
;;       )

;;; Customization:
;;
;;  There are many options to customize, too many to list here.  Each
;;  one is documented extensively, as are the groups they are in.  To
;;  customize this package do one of the following:
;;
;;     M-x customize-group win
;;
;;  or
;;
;;     M-x win-customize
;;
;;  Both of them do the same thing.

;;; Background:
;;
;;  win is yet another CUA emulator.  The two questions that beg
;;  to be answered immediately after reading that statement are:
;;
;;  1) What's CUA?
;;  2) Why does the world need another CUA emulator?
;;
;;  CUA stands for Common User Access, and it's the look-and-feel
;;  behind many common UI standards today, including Windows, Mac,
;;  Motif and others.  One of the many radical things that the Mac
;;  gave us way back when was standardization of keystrokes for
;;  commands like Open, Save and Print.  In the Emacs world, CUA means
;;  using keys that are familiar to many non-Emacs users, for common
;;  purposes.  So, for CUA people <Home> means beginning of line, and
;;  <Ctrl><A> means select all, while for Emacs people <Home> means
;;  beginning of buffer, and <Ctrl><A> means beginning of line.  Most
;;  Emacs users I know look down on CUA, but the lack of familiarity
;;  is what keeps most non-initiates from using Emacs, IMHO.
;;
;;  CUA also seems to imply (to many people) an alternate selection
;;  paradigm to the one that is native to Emacs.  Emacs has a
;;  point and a mark, and the way to kill and delete text (copy & cut,
;;  respectively, in CUA terms) is to set the mark, move point
;;  somewhere else, and kill or delete the text between them.  This
;;  means that you need to remember where you set mark, since by
;;  default there is no on-screen indicator.
;;
;;  Emacs natively gives you the tools to make the selection of text
;;  more CUA-like, by allowing the currently selected region to be
;;  highlighted on screen, which most CUA users will be familiar with.
;;  One of the hidden pitfalls, though, is that the region (the area
;;  between point and mark) is still defined even if it's not
;;  selected, which can lead to some strange behavior.  Fortunately,
;;  Emacs provides a way to disable this "feature", too.  Combining all
;;  of this with the ability to use the arrow keys (and other cursor
;;  movement keys) to select the text by holding down the shift key
;;  adds the final bit of CUA-like behavior to the mix.
;;
;;  So why does the world need another CUA emulator mode for Emacs?
;;  Because mine's better, of course!  Seriously though, I do believe
;;  that mine IS better, for me at least.  I've attempted to take the
;;  best of several other packages out there and roll it into
;;  something that's better than any of them individually.
;;
;;  The two major CUA emulator modes out there that I know of are
;;  CUA-mode and pc-select.  CUA-Mode is a rather heavy mode,
;;  providing many standard CUA functions and a whole lot more, and
;;  it's the one that I used for the first year-and-a-half of my Emacs
;;  experience.  They've solved the tricky problem of providing
;;  standard cut, copy & paste keys (<Ctrl><X>, <Ctrl><C> and
;;  <Ctrl><V> respectively), which is no small feat considering the
;;  amount of function bound to <Ctrl><X> and <Ctrl><C> by default.
;;  They allow both to cohabitate quite nicely, and only occasionally
;;  did I find myself trying to do something that required me to use
;;  one of their work-arounds.
;;
;;  After a while, though, I switched from CUA-mode to pc-select.  The
;;  major advantage that CUA-mode provides is the aforementioned key
;;  bindings for Cut, Copy and Paste, and I found myself seldom using
;;  them.  I normally use the alternative keystrokes for these
;;  (<Shift><Del> for cut, <Ctrl><Ins> for copy and <Shift><Ins> for
;;  paste), as they are available on nearly all of the platforms and
;;  programs I use, while the more standard CUA bindings are not.
;;  pc-select also provides a much better set of text
;;  selection/highlighting functions, which I found myself looking for
;;  eventually.  Soon thereafter, though, I started thinking about
;;  writing my own package.
;;
;;  There were two design goals driving the creation of this new
;;  package.  First, there needed to be a way to turn it off (as odd
;;  as that sounds).  I've grown tired of watching people who know
;;  Emacs sit down at my computer and proceed to select the entire
;;  buffer and delete it (you know, <Ctrl><A> to go to the beginning
;;  of the line and <Ctrl><X> as the prefix to some other command they
;;  want to run).  So, I wanted something that I can use that makes me
;;  more productive, but that I could turn off easily, so that others
;;  could sit down and help me when I needed it.  Alternately, I'd
;;  like to be able to sit down at someone else's computer and switch
;;  win on easily, do the work I need to, and then turn it back
;;  off.  To that end, it is very easy and non-intrusive to simply
;;  have win hanging around, waiting to be turned on when some
;;  fool like me sits down (see `win-bootstrap' for more
;;  details).
;;
;;  Secondly, I wanted something that was easily extendable,
;;  particularly as concerns text selection.  I'm a man who likes to
;;  tweak his editor, so I have key bindings for things that are
;;  outside the norm, and I wanted them to be able to extend the
;;  selection if I hold the shift key down, and to deactivate the
;;  selection if I don't.  Having switched from CUA-mode to pc-select
;;  helped me to implement this, but the pc-select package didn't make
;;  it easy enough IMHO.  That alone wasn't enough to write my own
;;  CUA-emulation mode, but combined with the first design goal, I now
;;  had a mission!

;;; To Do:
;;
;;  o Nothing, at the moment.

;;; Credits:
;;
;;  The selection code is conceptually based on the functions in
;;  pc-select.

;;; Comments:
;;
;;  Any comments, suggestions, bug reports or upgrade requests are welcome.
;;  Please send them to Joe Casadonte (emacs@northbound-train.com).
;;
;;  This version of win was developed and tested with NTEmacs
;;  20.7.1 under Windows 2000 & NT 4.0 and Emacs 20.7.1 under Linux
;;  (RH7).  Please, let me know if it works with other OS and versions
;;  of Emacs.

;;; Change Log:
;;
;;  see http://www.northbound-train.com/emacs/win.log

;;; **************************************************************************
;;; **************************************************************************
;;; **************************************************************************
;;; **************************************************************************
;;; **************************************************************************
;;; Code:

(eval-when-compile
  ;; silence the old byte-compiler
  (defvar byte-compile-dynamic)
  (set (make-local-variable 'byte-compile-dynamic) t)

  ;; silence lint
  (defvar minor-mode-alist)
  (defvar transient-mark-mode)
  (defvar mark-even-if-inactive)
  (defvar this-command))

;;; **************************************************************************
;;; ***** customization routines
;;; **************************************************************************
(defgroup win nil
  "Emulate CUA bindings."
  :group 'editing-basics
  :group 'convenience)

;; ---------------------------------------------------------------------------
(defun win-customize ()
  "Customization of the group 'win'."
  (interactive)
  (customize-group "win"))

;; ---------------------------------------------------------------------------
(defcustom win-display-status-in-mode-line t
  "Used to show or hide mode-line indicator."
  :type 'boolean
  :group 'win)

;; ---------------------------------------------------------------------------
(defcustom win-mode-line-string " CUA"
  "String to display in mode-line when 'win' is active."
  :type 'string
  :group 'win
  :set (lambda (symbol newval)
		 (setq win-mode-line-string newval)
		 (let ((cell (assoc 'win minor-mode-alist)))
		   (when cell
			 (setcdr cell (list win-mode-line-string))
			 (force-mode-line-update)))))

;; ---------------------------------------------------------------------------
(defcustom win-default-keybindings 3
  "Select which keys are bound when 'win' is activated.

If nil, other CUA keys will be bound -- see function `win' for
more details."
  :type '(choice (const :tag "Movement keys only" 1)
				 (const :tag "Non-movement keys only" 2)
				 (other :tag "Both movement & non-movement-keys" 3))

  :group 'win)

;; ---------------------------------------------------------------------------
;; ---------------------------------------------------------------------------
(defgroup win-basic-options nil
  "Toggle use of many basic CUA or CUA-like options.

In general, the default value reflects what I think most people would
consider as normal behavior in a CUA environment.  If the default
value is nil, the implication is that I think this is a useful
extension worthy of your consideration.  If the default is t, then
this is something I think people may have reason to want to turn off."
  :group 'win)

;; ---------------------------------------------------------------------------
(defcustom win-ignore-key-list ()
  "List of non-movement keys to ignore when enabling win.

The value for this should be the string representation of the key to
ignore.  For example, to instruct win NOT to bind C-w to
`kill-buffer' you would add 'C-w' (without the quotes) to this list.

NOTE: this is for non-movement keys only (see function `win' for
more details on what is a movement key and what is a non-movement
key).  Optional movement keys are customizable individually or in
pairs, as they generally require more explanation then the
non-movement keys."
  :type '(repeat string)
  :group 'win-basic-options)

;; ---------------------------------------------------------------------------
(defcustom win-keep-current-mark nil
  "Determines whether or not mark is cleared when 'win' is enabled.

t - current mark is kept
nil - current mark is NOT kept

If the current mark is kept when 'win' is enabled then there will
be a selection active when the mode is first enabled, which is
probably not the desired behavior.  If this optional is nil and for
whatever reason you DO wish to activate the previous mark, typing
\\[exchange-point-and-mark] twice will do the trick nicely."
  :type 'boolean
  :group 'win-basic-options)

;; ---------------------------------------------------------------------------
(defcustom win-use-home-key-bounce nil
  "Cause 'home' to bounce between column zero and the first non-blank column.

t - use home key bounce
nil - do NOT use home key bounce

If the user presses 'home' when in any column other than zero, point
is placed in column zero.  If it's pressed when in column zero, point
moves to the first non-whitespace column (if there is one)."
  :type 'boolean
  :group 'win-basic-options)

;; ---------------------------------------------------------------------------
(defcustom win-use-end-key-bounce nil
  "Cause 'end' to bounce between the last column and the last non-blank column.

t - use end key bounce
nil - do NOT use end key bounce

If the user presses 'end' when in any column other than the last
column in the line, point is placed in the last column.  If it's
pressed when in the last column, point moves to the last
non-whitespace column (if there is one)."
  :type 'boolean
  :group 'win-basic-options)

;; ---------------------------------------------------------------------------
(defcustom win-disable-inactive-regions t
  "Deactivate the region when nothing is explicity selected/highlighted.

t - disable inactive regions
nil - do NOT disable inactive regions

See the variable `mark-even-if-inactive' for more information."
  :type 'boolean
  :group 'win-basic-options)

;; ---------------------------------------------------------------------------
(defcustom win-use-simplified-paragraph-movement nil
  "Use simplified definitions of `forward-paragraph' and `backward-paragraph'.

t - use the simplified functions
nil - use the normal functions

The normal versions of `forward-paragraph' and `backward-paragraph'
use a mode-specific definition of what a paragraph is.  The simplified
versions just look for one or more empty lines to distinguish a
paragraph.

See also `win-rebind-normal-paragraph-movement-fns'."
  :type 'boolean
  :group 'win-basic-options)

;; ---------------------------------------------------------------------------
(defcustom win-rebind-normal-paragraph-movement-fns nil
  "Bind normal (mode-specific) paragraph movement functions to alternate keys.

t - re-bind the normal functions (see below)
nil - do not re-bind the normal functions to anything

If true, re-binds `forward-paragraph' and `backward-paragraph' to
'M-up' and 'M-down', respectively.  This variable has no effect if
`win-use-simplified-paragraph-movement' is not true."
  :type 'boolean
  :group 'win-basic-options)

;; ---------------------------------------------------------------------------
(defcustom win-use-page-movement-keys t
  "Bind 'C-prior' and 'C-next' to top- and bottom-of-page, respectively.

t - bind the keys
nil - do not bind the keys

top-of-page moves point to the top line of the current window without
scrolling.  Similarly, bottom-of-page moves point to the last line of
the current window without scrolling."
  :type 'boolean
  :group 'win-basic-options)

;; ---------------------------------------------------------------------------
(defcustom win-use-backward-delete-word t
  "Binds 'C-backspace' to `win-backward-delete-word'.

t - use cua-backward-delete-word
nil - do not use cua-backward-delete-word

This is like `backward-kill-word' except that the word is deleted, not
killed (i.e. it is not saved to the kill-ring/clipboard)."
  :type 'boolean
  :group 'win-basic-options)

;; ---------------------------------------------------------------------------
;; ---------------------------------------------------------------------------
(defgroup win-third-party-packages nil
  "Toggle use of third-party packages in win mode.

In general, if the package ships with Emacs it's turned on by default,
and if not, it's turned off.  Simply by being listed here, the implication
is that I recommend its use."
  :group 'win)

;; ---------------------------------------------------------------------------
(defcustom win-use-pager-fns t
  "Toggles use of the 'pager' package (which-see).

Latest version is available at:

	http://www.docs.uu.se/~mic/emacs.html"
  :type 'boolean
  :group 'win-third-party-packages)

;; ---------------------------------------------------------------------------
(defcustom win-use-hscroll-mode t
  "Toggles use of the 'hscroll' package (which-see).

This option is ignored in Emacs 21 (which has horizontal scrolling
built in)."
  :type 'boolean
  :group 'win-third-party-packages)

;; ---------------------------------------------------------------------------
(defcustom win-use-delete-selection-mode t
  "Toggles use of the 'delsel' package (which-see)."
  :type 'boolean
  :group 'win-third-party-packages)

;; ---------------------------------------------------------------------------
(defcustom win-use-blinking-cursor nil
  "Toggles use of the 'blinking-cursor' package (which-see).

Requires third-party package not normally shipped with Emacs.  It is
available at:

	http://www.wonderworks.com

This option is ignored in Emacs 21 (which has cursor blinking built
in).  The 'blinking-cursor' package provides methods for setting the
blink-rate and blink colors (which see)."
  :type 'boolean
  :group 'win-third-party-packages)

;; ---------------------------------------------------------------------------
(defcustom win-use-bar-cursor nil
  "Toggles use of the 'bar-cursor' package (which-see).

Requires third-party package not normally shipped with Emacs.  It is
available at:

	http://www.northbound-train.com/emacs.html"
  :type 'boolean
  :group 'win-third-party-packages)

;; ---------------------------------------------------------------------------
;; ---------------------------------------------------------------------------
(defgroup win-disable-effects nil
  "Controls what happens when the mode is toggled off.

By default, most win options turn off when win does."
  :group 'win)

;; ---------------------------------------------------------------------------
(defcustom win-retain-pager-fns t
  "Controls whether or not 'pager' is turned off when 'win' is disabled.

t - pager is NOT turned off
nil - pager IS turned off

This variable has no effect if `win-use-pager-fns' is not true."
  :type 'boolean
  :group 'win-disable-effects)

;; ---------------------------------------------------------------------------
(defcustom win-retain-hscroll-mode t
  "Controls if 'hscroll-mode' is turned off when 'win' is disabled.

t - 'hscroll-mode' is NOT turned off
nil - 'hscroll-mode' IS turned off

This variable has no effect if `win-use-hscroll-mode' is not true."
  :type 'boolean
  :group 'win-disable-effects)

;; ---------------------------------------------------------------------------
(defcustom win-retain-delete-selection-mode t
  "Controls if 'delete-selection-mode' is active when 'win' is disabled.

t - 'delete-selection-mode' is NOT turned off
nil - 'delete-selection-mode' IS turned off

This variable has no effect if `win-use-delete-selection' is not true.
See command `delete-selection-mode' for more information."
  :type 'boolean
  :group 'win-disable-effects)

;; ---------------------------------------------------------------------------
(defcustom win-retain-blinking-cursor nil
  "Controls if 'blinking-cursor-mode' is turned off when 'win' is disabled.

t - 'blinking-cursor-mode' is NOT turned off
nil - 'blinking-cursor-mode' IS turned off

This variable has no effect if `win-use-blinking-cursor' is not true.
See command `blinking-cursor-mode' for more information."
  :type 'boolean
  :group 'win-disable-effects)

;; ---------------------------------------------------------------------------
(defcustom win-retain-bar-cursor nil
  "Controls if 'bar-cursor-mode' is turned off when 'win' is disabled.

t - 'bar-cursor-mode' is NOT turned off
nil - 'bar-cursor-mode' IS turned off

This variable has no effect if `win-use-bar-cursor' is not true.
See command `bar-cursor-mode' for more information."
  :type 'boolean
  :group 'win-disable-effects)

;; ---------------------------------------------------------------------------
(defcustom win-retain-transient-mark nil
  "Controls if 'transient-mark' is turned off when 'win' is disabled.

t - 'transient-mark-mode' is NOT turned off
nil - 'transient-mark-mode' IS turned off

See the variable `transient-mark-mode' for more information."
  :type 'boolean
  :group 'win-disable-effects)

;; ---------------------------------------------------------------------------
(defcustom win-retain-suppress-inactive-regions nil
  "Controls if inactive regions are suppressed when 'win' is disabled.

t - inactive regions are still suppressed
nil - inactive regions are NOT suppressed

This variable has no effect if `win-disable-inactive-regions' is not true.
See the variable `mark-even-if-inactive' for more information."
  :type 'boolean
  :group 'win-disable-effects)

;; ---------------------------------------------------------------------------
;; ---------------------------------------------------------------------------
(defgroup win-hooks nil
  "Hooks for use in win mode."
  :group 'win)

;; ---------------------------------------------------------------------------
(defcustom win-load-hook nil
  "Hook to run when package is loaded."
  :type 'hook
  :group 'win-hooks)

;; ---------------------------------------------------------------------------
(defcustom win-hook nil
  "Hook called when 'win' mode is toggled.

Hook is run before ON or OFF hooks are run."
  :type 'hook
  :group 'win-hooks)

;; ---------------------------------------------------------------------------
(defcustom win-on-hook nil
  "Hook called when 'win' mode is turned on.

Hook is run after all other enable actions are taken."
  :type 'hook
  :group 'win-hooks)

;; ---------------------------------------------------------------------------
(defcustom win-off-hook nil
  "Hook called when 'win' mode is turned off.

Hook is run after all other disable actions are taken."
  :type 'hook
  :group 'win-hooks)

;; ---------------------------------------------------------------------------
(defcustom win-bind-keys-hook nil
  "Hook called when 'win' keys are bound.

The hook is called after all 'win' keys are bound."
  :type 'hook
  :group 'win-hooks)

;;; **************************************************************************
;;; ***** version related routines
;;; **************************************************************************
(defconst win-version
  "$Revision: 1.3 $"
  "Version number for 'win' package.")

;; ---------------------------------------------------------------------------
(defun win-version-number ()
  "Return 'win' version number."
  (string-match "[0123456789.]+" win-version)
  (match-string 0 win-version))

;; ---------------------------------------------------------------------------
(defun win-display-version ()
  "Display 'win' version."
  (interactive)
  (message "win version <%s>." (win-version-number)))

;;; **************************************************************************
;;; ***** constants
;;; **************************************************************************
(defconst win-is-emacs-21
  (if (string-match "Emacs 21" (emacs-version)) t nil)
  "Are we running 'win' under Emacs 21 or not?")

;;; **************************************************************************
;;; ***** mini-mode setup
;;; **************************************************************************
(defvar win nil "Non-nil if 'win' is enabled.")

;; ---------------------------------------------------------------------------
(defvar win-key-restoration-list ()
  "Used to store current key bindings to aid in restoration of key bindings.")

;; ---------------------------------------------------------------------------
;;;###autoload
(defun win (&optional arg)
  "Light-weight CUA emulator that is expandable and toggle-able.

Optional ARG turns mode on iff ARG is a positive integer.  If mode is
already in the state requested, no actions are taken.

When enabled, the following keys are bound (some are optional):
  Movement keys:
    o <left>, <right>, <up>, <down> - standard definitions
    o C-<right>, C-<left> - forward & backward word
    o C-<up>, C-<down> - forward & backward paragraph
    o M-<up>, M-<down> - forward & backward paragraph
      see `win-use-simplified-paragraph-movement'
    o <prior>, <next> - page up & down (scroll down & up, respectively)
    o C-<prior>, C-<next> - top & bottom of page
      see `win-use-page-movement-keys'
    o <home>, <end> - beginning & end of line
      see `win-use-home-key-bounce' & `win-use-end-key-bounce'
    o C-<home>, C-<end> - beginning & end of buffer

  Non-movement keys:
    o C-<backspace> - delete word backward
      see `win-use-backward-delete-word'
    o C-a - Select All (`mark-whole-buffer')
    o C-f - Find (`isearch-forward')
    o M-f - Find Backwards (`isearch-backward')
    o C-o - Open (`find-file')
    o C-r - Replace (`replace-string')
    o C-s - Save (`save-buffer')
    o M-s - Save As (`write-file')
    o C-w - Close (`kill-buffer')
    o C-z - Undo (`undo' / `advertised-undo')

In addition, certain CUA-like features of Emacs are enabled.
This is accomplished by modifying the values of the following
variables or calling the following functions (which see):
    o `mark-active' - see `win-keep-current-mark'
    o `truncate-lines' - see `win-use-hscroll-mode'
    o `hscroll-global-mode' - see `win-use-hscroll-mode'
    o `blinking-cursor-mode' - see `win-use-blinking-cursor'
    o `bar-cursor-mode' - see `win-use-bar-cursor-mode'
    o `transient-mark-mode'
    o `mark-even-if-inactive' - see `win-disable-inactive-regions'
    o `delete-selection-mode' - see `win-use-delete-selection-mode'"
  (interactive "P")

  ;; toggle on and off
  (let ((old-mode win))
	(setq win
		  (if arg (or (listp arg)
					  (> (prefix-numeric-value arg) 0))
			(not win)))

	;; nothing is ever done unless we're actually changing modes
	(if (not (equal old-mode win))
		(progn
		  ;; general hook
		  (if win-hook
			  (run-hooks 'win-hook))

		  ;; off or on?
		  (if win
			  (win-internal-enable-cmd)
			(win-internal-disable-cmd))

		  ;; we're done
		  (when (interactive-p)
			(message "win %s." (if win "enabled" "disabled"))))

	  ;; signal an error
	  (when (interactive-p)
		(error "Package 'win' is already %s" (if win "enabled" "disabled"))))
	))

;; ---------------------------------------------------------------------------
(defun win-internal-enable-cmd ()
  "Internal function called when 'win' is enabled."

  ;; (re-)bind CUA keys
  (win-reset-keybindings)

  ;; some autoloads
  (when win-use-hscroll-mode
	(autoload 'hscroll-global-mode "hscroll"))

  (when win-use-pager-fns
	(autoload 'pager-page-up "pager")
	(autoload 'pager-page-down "pager"))

  (when win-use-blinking-cursor
	(autoload 'blinking-cursor-mode "blinking-cursor"))

  (when win-use-bar-cursor
	(autoload 'bar-cursor-mode "bar-cursor"))

  ;; get rid of current mark (if any)
  (unless win-keep-current-mark
	(setq mark-active nil))

  ;; other CUA-like things
  (when (and win-use-hscroll-mode
			 (not win-is-emacs-21))
	(setq-default truncate-lines t)
	(hscroll-global-mode 1))

  (when (and win-use-blinking-cursor
			 (not win-is-emacs-21))
	(blinking-cursor-mode 1))

  (when win-use-bar-cursor
	(bar-cursor-mode 1))

  ;; transient mode mark
  (setq transient-mark-mode t)

  ;; inactive mark
  (when win-disable-inactive-regions
	(setq mark-even-if-inactive nil))

  ;; delete selection mode
  (when win-use-delete-selection-mode
	(delete-selection-mode 1))

  ;; hook when turning on
  (when win-on-hook
	(run-hooks 'win-on-hook)))

;; ---------------------------------------------------------------------------
(defun win-internal-disable-cmd ()
  "External function called when 'win' is disabled."

  ;; restore old keymap
  (win-restore-orig-keys)

  ;; disable pager fns maybe
  (when (and win-use-pager-fns
			 win-retain-pager-fns)
	(global-set-key (kbd "<prior>") 'pager-page-up)
	(global-set-key (kbd "<next>") 'pager-page-down))

  ;; turn off hscroll mode maybe
  (when (and win-use-hscroll-mode
			 (not win-retain-hscroll-mode)
			 (not win-is-emacs-21))
	(hscroll-global-mode nil))

  ;; turn off blinking cursor mode maybe
  (when (and win-use-blinking-cursor
			 (not win-retain-blinking-cursor)
			 (not win-is-emacs-21))
	(blinking-cursor-mode 0))

  ;; turn off bar cursor mode maybe
  (when (and win-use-bar-cursor
			 (not win-retain-bar-cursor))
	(bar-cursor-mode 0))

  ;; turn off transient mark mode, maybe
  (when (not win-retain-transient-mark)
	(setq transient-mark-mode nil))

  ;; turn off disable inactive regions, maybe
  (when (and win-disable-inactive-regions
			 (not win-retain-suppress-inactive-regions))
	(setq mark-even-if-inactive t))

  ;; turn off delete selection mode, maybe
  (when (and win-use-delete-selection-mode
			 (not win-retain-delete-selection-mode))
	(delete-selection-mode 0))

  ;; hook when turning off
  (when win-off-hook
	(run-hooks 'win-off-hook)))

;; ---------------------------------------------------------------------------
;; add to minor-mode-alist if not there already
(when win-display-status-in-mode-line
  (or
   (assq 'win minor-mode-alist)
   (setq minor-mode-alist
		 (cons
		  (list 'win win-mode-line-string)
		  minor-mode-alist))))

;;; **************************************************************************
;;; ***** key-binding fns
;;; **************************************************************************
(defun win-save-keystroke-for-restoration (keystroke)
  "Save KEYSTROKE for possible restoration if 'win' is ever disabled."
  (setq win-key-restoration-list
		(append (vector (list keystroke (key-binding (read-kbd-macro keystroke))))
				win-key-restoration-list)))

;; ---------------------------------------------------------------------------
(defun win-bind-key-simple (keystroke fn &optional ignore-list)
  "Save current binding for KEYSTROKE and set new binding to FN.

KEYSTROKE should be a string suitable to be passed into `read-kbd-macro'.

Current binding of KEYSTROKE is saved for possible restoration later
if 'win' is ever disabled.

IGNORE-LIST is a list of keys to ignore.  Interally, the value of
`win-ignore-key-list' is passed in, allowing you to selectively
disable individual keybindings."
  ;; check ignore list
  (unless (member keystroke win-ignore-key-list)
	;; save it for later restoration
	(win-save-keystroke-for-restoration keystroke)
	(global-set-key (read-kbd-macro keystroke) fn)))

;; ---------------------------------------------------------------------------
(defun win-bind-both-motion-keys (keystroke fn &optional imitate)
  "Call `win-bind-motion-key' for KEYSTROKE & S-KEYSTROKE.

`win-bind-motion-key' is called once for KEYSTROKE and again
for Shift-KEYSTROKE, with ACT-MARK values of nil and t, respectively.
See `win-bind-motion-key' for more details on KEYSTROK, FN and
IMITATE."
  (let ((shifted (concat "S-" keystroke)))
	(win-bind-motion-key keystroke fn nil imitate)
	(win-bind-motion-key shifted fn t imitate))
  )

;; ---------------------------------------------------------------------------
(defun win-bind-motion-key (keystroke fn act-mark &optional imitate read-only)
  "Save current binding for KEYSTROKE and set new binding to FN.

KEYSTROKE should be a string suitable to be passed into `read-kbd-macro'.

Current binding of KEYSTROKE is saved for possible restoration later
if 'win' is ever disabled.

If ACT-MARK is nil, KEYSTROKE is bound to a lambda expression that
deactivates mark and calls FN.  If ACT-MARK is non-nil, KEYSTROKE is
bound to a lambda expression that activates mark and calls FN.  In
either case, the original FN can be imitated by passing in a non-nil
value to IMITATE.  Finally, if READ-ONLY is non-nil, the `interactive'
form of the lambda expression will be set to \"*p\", otherwise it will
be set to just \"p\"."
  (let ((doc-string (concat (if act-mark "Activate" "Deactivate")
							" mark and call `" (symbol-name fn) "' interactively."))
		(interactive-string (if read-only "*p" "p")))

	;; save current definition for possible later restoration
	(win-save-keystroke-for-restoration keystroke)

	;; bind the new key
	(global-set-key (read-kbd-macro keystroke)
					`(lambda (prefix)
					   ,doc-string
					   (interactive ,interactive-string)
					   (win-ensure-mark ,act-mark)
					   (call-interactively ',fn)
					   (when ,imitate
						 (setq this-command ',fn)))
					)
	))

;; ---------------------------------------------------------------------------
(defun win-restore-orig-keys ()
  "Restore original key-bindings of all keys bound thru the 'win' package."
  (mapcar (lambda (binding)
			(let ((keystroke (car binding))
				  (fn (cadr binding)))
			  (if fn
				  (global-set-key (read-kbd-macro keystroke) fn)
				(global-unset-key (read-kbd-macro keystroke)))))
		  win-key-restoration-list))

;;; **************************************************************************
;;; ***** util functions
;;; **************************************************************************
(defun win-ensure-mark (activate)
  "Ensures that mark is in the desired state.

If ACTIVATE is nil, mark will be turned off (if it's not off already).
if ACTIVATE is non-nil, mark will be activated if it's not already."
  (if activate
	  (or mark-active (set-mark-command nil))
	(setq mark-active nil)))

;; ---------------------------------------------------------------------------
(defun win-reset-keybindings ()
  "Reset current global keymap to CUA bindings."
  ;; clear out old restoration list
  (setq win-key-restoration-list '())

  ;; bind standard motion keys
  (unless (= win-default-keybindings 2)
	(win-bind-both-motion-keys "<left>" 'backward-char)
	(win-bind-both-motion-keys "<right>" 'forward-char)
	(win-bind-both-motion-keys "<up>" 'previous-line t)
	(win-bind-both-motion-keys "<down>" 'next-line t)

	(win-bind-both-motion-keys "C-<right>" 'forward-word)
	(win-bind-both-motion-keys "C-<left>" 'backward-word)

	(if win-use-simplified-paragraph-movement
		(progn
		  (win-bind-both-motion-keys "C-<up>" 'win-simple-backward-paragraph)
		  (win-bind-both-motion-keys "C-<down>" 'win-simple-forward-paragraph)
		  (when win-rebind-normal-paragraph-movement-fns
			(win-bind-both-motion-keys "M-<up>" 'backward-paragraph)
			(win-bind-both-motion-keys "M-<down>" 'forward-paragraph)))
	  (win-bind-both-motion-keys "C-<up>" 'backward-paragraph)
	  (win-bind-both-motion-keys "C-<down>" 'forward-paragraph))

	(when win-use-page-movement-keys
	  (win-bind-both-motion-keys "C-<prior>" 'win-top-of-page)
	  (win-bind-both-motion-keys "C-<next>" 'win-bottom-of-page))

	(if win-use-home-key-bounce
		(win-bind-both-motion-keys "<home>" 'win-home-key-bounce)
	  (win-bind-both-motion-keys "<home>" 'beginning-of-line))
	(if win-use-end-key-bounce
		(win-bind-both-motion-keys "<end>" 'win-end-key-bounce)
	  (win-bind-both-motion-keys "<end>" 'end-of-line t))

	(win-bind-both-motion-keys "C-<home>" 'beginning-of-buffer)
	(win-bind-both-motion-keys "C-<end>" 'end-of-buffer)

	(if win-use-pager-fns
		(progn
		  (win-bind-both-motion-keys "<prior>" 'pager-page-up)
		  (win-bind-both-motion-keys "<next>" 'pager-page-down))
	  (win-bind-both-motion-keys "<prior>" 'scroll-down)
	  (win-bind-both-motion-keys "<next>" 'scroll-up))
	)

  (unless (= win-default-keybindings 1)
	(when win-use-backward-delete-word
	  (win-bind-motion-key "C-<backspace>" 'win-backward-delete-word nil nil t))

	;; bind CUA-esque keys
	(win-bind-key-simple "C-a" 'mark-whole-buffer)
	(win-bind-key-simple "C-f" 'isearch-forward)
	(win-bind-key-simple "M-f" 'isearch-backward)
	(win-bind-key-simple "C-o" 'find-file)
	(win-bind-key-simple "C-r" 'replace-string)
	(win-bind-key-simple "C-s" 'save-buffer)
	(win-bind-key-simple "M-s" 'write-file)
	(win-bind-key-simple "C-w" 'kill-buffer)
	(win-bind-key-simple "C-z" 'undo)
	)

  ;; finally, call hook
  (run-hooks 'win-bind-keys-hook)
  )

;; ---------------------------------------------------------------------------
(defun win-home-key-bounce ()
  "Causes point to alternate between column 0 & the first non-blank column.

See `win-use-home-key-bounce' for more information."
  (interactive)
  (let ((bolp (bolp))
		(orig (point)))
	(beginning-of-line)
	(if (and bolp (bolp))
		(let ((eol (line-end-position)))
		  (skip-chars-forward " \t\n" eol)
		  (if (and (= eol (point))
				   (/= eol orig))
			  (beginning-of-line))
		  ))))

;; ---------------------------------------------------------------------------
(defun win-end-key-bounce ()
  "Cause point to alternate between the last column & the last non-blank column.

See `win-use-end-key-bounce' for more information."
  (interactive)
  (let ((eolp (eolp))
		(orig (point)))
	(end-of-line)
	(if (and eolp (eolp))
		(let ((bol (line-beginning-position)))
		  (skip-chars-backward " \t\n" bol)
		  (if (and (= bol (point))
				   (/= bol orig))
			  (end-of-line))
		  )))

  ;; imitate end-of-line
  (setq this-command 'end-of-line))

;; ---------------------------------------------------------------------------
(defun win-top-of-page ()
  "Causes point to move to the top of the current window.

See `win-use-page-movement-keys' for more information."
  (interactive)
  (move-to-window-line 0))

;; ---------------------------------------------------------------------------
(defun win-bottom-of-page ()
  "Causes point to move to the bottom of the current window.

See `win-use-page-movement-keys' for more information."
  (interactive)
  (move-to-window-line -1))

;; ---------------------------------------------------------------------------
(defun win-simple-forward-paragraph (prefix)
  "Move point forward to the end of the next paragraph.

With prefix argument PREFIX, do this that many times.

See `win-use-simplified-paragraph-movement' for more
information."
  (interactive "p")
  (save-match-data

	(let ((lcv 0))
	  ;; do N times
	  (while (< lcv prefix)
		;; go forward to next non-empty line
		(re-search-forward "^\\s-*\\S-+" (point-max) 1)

		;; and then find the next empty line
		(re-search-forward "^\\s-*$" (point-max) 1)

		(setq lcv (1+ lcv))))))

;; ---------------------------------------------------------------------------
(defun win-simple-backward-paragraph (prefix)
  "Move point backward to the beginning of the previous paragraph.

With prefix argument PREFIX, do this that many times.

See `win-use-simplified-paragraph-movement' for more
information."
  (interactive "p")
  (save-match-data

	(let ((lcv 0))
	  (beginning-of-line)

	  ;; do N times
	  (while (< lcv prefix)
		;; go backward to next non-empty line
		(while (looking-at "^\\s-*$")
		  (previous-line 1))

		;; and then find the next (previous) empty line
		(re-search-backward "^\\s-*$" (point-min) 1)
		(setq lcv (1+ lcv))))))

;; ---------------------------------------------------------------------------
(defun win-delete-word (arg)
  "Delete characters forward until encountering the end of a word.

With prefix argument ARG, do this that many times."
  (interactive "p")
  (delete-region (point) (progn (forward-word arg) (point))))

;; ---------------------------------------------------------------------------
(defun win-backward-delete-word (arg)
  "Delete characters backward until encountering the end of a word.

With prefix argument ARG, do this that many times."
  (interactive "p")
  (win-delete-word (- arg)))

;;; **************************************************************************
;;; ***** we're done
;;; **************************************************************************
(provide 'win)
(run-hooks 'win-load-hook)

;;; win.el ends here
;;; **************************************************************************
;;;; *****  EOF  *****  EOF  *****  EOF  *****  EOF  *****  EOF  *************



